<?php

namespace App\Http\Controllers;

use App\Models\Prescription;
use App\Models\PrescriptionItem;
use App\Models\Appointment;
use App\Models\User;
use App\Services\PrescriptionPdfService;
use App\Notifications\PrescriptionCreatedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class PrescriptionController extends Controller
{
    protected $pdfService;

    public function __construct(PrescriptionPdfService $pdfService)
    {
        $this->pdfService = $pdfService;
    }

    public function create($appointmentId)
    {
        $appointment = Appointment::with(['patient', 'doctor', 'vitals'])->findOrFail($appointmentId);
        
        if (!Auth::user()->isDoctor() || $appointment->doctor_id !== Auth::user()->doctor->id) {
            abort(403);
        }

        return view('prescriptions.create', compact('appointment'));
    }

    public function store(Request $request, $appointmentId)
    {
        $appointment = Appointment::findOrFail($appointmentId);
        $doctor = Auth::user()->doctor;

        $validated = $request->validate([
            'diagnosis' => 'nullable|string',
            'additional_prescription_details' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.medicine_name' => 'required|string',
            'items.*.dosage' => 'required|string',
            'items.*.frequency' => 'required|string',
            'items.*.instructions' => 'nullable|string',
            'items.*.duration' => 'required|string',
        ]);

        $prescription = Prescription::create([
            'appointment_id' => $appointment->id,
            'doctor_id' => $doctor->id,
            'patient_id' => $appointment->patient_id,
            'prescription_number' => 'RX' . strtoupper(Str::random(10)),
            'diagnosis' => $validated['diagnosis'],
            'additional_prescription_details' => $validated['additional_prescription_details'],
        ]);

        // Reload appointment with pharmacist relationship
        $appointment->load('pharmacist');

        foreach ($validated['items'] as $index => $item) {
            PrescriptionItem::create([
                'prescription_id' => $prescription->id,
                'item_number' => $index + 1,
                'medicine_name' => $item['medicine_name'],
                'dosage' => $item['dosage'],
                'frequency' => $item['frequency'],
                'instructions' => $item['instructions'] ?? null,
                'duration' => $item['duration'],
            ]);
        }

        // Generate PDF
        $pdfPath = $this->pdfService->generate($prescription);
        $prescription->update(['pdf_path' => $pdfPath]);

        // Update appointment status
        $appointment->update(['status' => 'completed']);

        // Notify pharmacist if appointment has pharmacist
        if ($appointment->pharmacist_id && $appointment->pharmacist) {
            $pharmacistUser = $appointment->pharmacist->user;
            if ($pharmacistUser) {
                try {
                    $pharmacistUser->notify(new PrescriptionCreatedNotification($prescription));
                } catch (\Exception $e) {
                    // Log error but continue - database notification will still work
                    \Log::warning('Failed to send email notification to pharmacist: ' . $e->getMessage());
                }
            }
        }

        // Notify all admins
        $admins = User::where('role', 'admin')->where('is_active', true)->get();
        foreach ($admins as $admin) {
            try {
                $admin->notify(new PrescriptionCreatedNotification($prescription));
            } catch (\Exception $e) {
                // Log error but continue - database notification will still work
                \Log::warning('Failed to send email notification to admin: ' . $e->getMessage());
            }
        }

        return redirect()->route('doctor.prescriptions')
            ->with('success', 'Prescription created successfully.');
    }

    public function show($id)
    {
        $prescription = Prescription::with(['patient', 'doctor', 'items', 'appointment'])->findOrFail($id);
        return view('prescriptions.show', compact('prescription'));
    }

    public function download($id)
    {
        $prescription = Prescription::findOrFail($id);
        
        if (!$prescription->pdf_path || !file_exists(storage_path('app/public/' . $prescription->pdf_path))) {
            $pdfPath = $this->pdfService->generate($prescription);
            $prescription->update(['pdf_path' => $pdfPath]);
        }

        return response()->download(storage_path('app/public/' . $prescription->pdf_path));
    }
}

